// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Remix_RGBA.A.fsh                               ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.2 (2017/04/02)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To modify the colours of an image by remixing the red, green, ==
// == blue and alpha components of each pixel.                               ==
// ==                                                                        ==
// == Description: Each component, Red, Green, Blue and Alpha, of the source ==
// == image is modified using                                                ==
// ==                                                                        ==
// ==    C.r' = C.r + K.r + C.r*R.r + C.g*R.g + C.b*R.b + C.a*R.a            ==
// ==    C.g' = C.g + K.g + C.r*G.r + C.g*G.g + C.b*G.b + C.a*G.a            ==
// ==    C.b' = C.b + K.b + C.r*B.r + C.g*B.g + C.b*B.b + C.a*B.a            ==
// ==    C.a' = C.a + K.a + C.r*A.r + C.g*A.g + C.b*A.b + C.a*A.a            ==
// ==                                                                        ==
// == or more succinctly                                                     ==
// ==                                                                        ==
// ==    C.r' = C.r + K.r + dot(C,R)                                         ==
// ==    C.g' = C.g + K.g + dot(C,G)                                         ==
// ==    C.b' = C.b + K.b + dot(C,B)                                         ==
// ==    C.a' = C.a + K.a + dot(C,A)                                         ==
// ==                                                                        ==
// == or even more succinctly as                                             ==
// ==                                                                        ==
// ==    C' = C + K + C*M                                                    ==
// ==                                                                        ==
// == where C is the source colour, R, G, B, A, K four element vectors and M ==
// == the 4x4 matrix with R, G, B and A as its rows.                         ==
// ==                                                                        ==
// == The parameters K, R, G, B and A are supplied as uniform variables that ==
// == may be defined in .scn files using the shader. All default to zero.    ==
// ==                                                                        ==
// == The shader can be used to achieve several colour effects. e.g.         ==
// ==                                                                        ==
// ==   Remove one or more colour components                                 ==
// ==   Invert one or more colour components                                 ==
// ==   Reduce or enhance one or more components                             ==
// ==   Average over two or more colour components                           ==
// ==   Rotate about the luminosity axis in colour space.                    ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/03/27 - v1.0.0 - Initial version.                               ==
// ==   2017/03/28 - v1.0.1 - Added clamping and luminosity options.         ==
// ==   2017/04/02 - v1.0.2 - Changed to using an explicit matrix form.      ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

vec2 Emu_Normalise_to_Window ( vec2 xy ) { return xy / u_WindowSize.xy; }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters.

uniform vec2 Emu_Remix_RGBA_scale;
uniform vec2 Emu_Remix_RGBA_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the output of the shader is calculated as described above with
// no attempt being made to ensure that the calculated colour components are
// within their valid range of 0.0 to 1.0. The mode control parameter can be
// used to modify the result to ensure that it is valid. The parameter takes
// the form of a two digit integer where the more significant digit is  used
// to control clamping and the less significant digit to control luminosity.
//
// Valid values for the clamp mode, i.e. the more significant digit, are
//
//   0 - default - no clamping
//   1 - clamp R, G, B and A
//   2 - clamp R, G and B
//   3 - clamp A only
//
// and those for the luminosity mode, the less significant digit, are
//
//   0 - default - no luminosity correction
//   1 - luminosity preserved

uniform int Emu_Remix_RGBA_mode;

int clamping_mode   = int ( mod(Emu_Remix_RGBA_mode/10.0,10.0) );
int luminosity_mode = int ( mod(Emu_Remix_RGBA_mode,10.0) );

bool constant_luminosity = luminosity_mode == 1;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The colour mixing parameters.  See the overall description of this shader,
// above, for details of what each of these are.

uniform vec4 Emu_Remix_RGBA_R; // Mixing parameters for the Red channel
uniform vec4 Emu_Remix_RGBA_G; // Mixing parameters for the Green channel
uniform vec4 Emu_Remix_RGBA_B; // Mixing parameters for the Blue channel
uniform vec4 Emu_Remix_RGBA_A; // Mixing parameters for the Alpha channel
uniform vec4 Emu_Remix_RGBA_K; // Additive terms for the R,G,B,A channels

#define R Emu_Remix_RGBA_R
#define G Emu_Remix_RGBA_G
#define B Emu_Remix_RGBA_B
#define A Emu_Remix_RGBA_A
#define K Emu_Remix_RGBA_K

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Determine the lower and upper limits to which the colour channels will be
// clamped.  If clamping is not enabled then the limits used are effectively
// infinite.

bool clamp_RGB = ( clamping_mode == 1 ) || ( clamping_mode == 2 );
bool clamp_A   = ( clamping_mode == 1 ) || ( clamping_mode == 3 );
vec4 clamp_X   = vec4 ( vec3(float(clamp_RGB)), clamp_A );

vec4 clamp_low  = -1.0e32 * clamp_X;
vec4 clamp_high = 1.0 - clamp_low;

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_Remix_RGBA ( vec2 uv )
 {
   // Get and manipulate the colour and opacity for the current point.

   vec4 c = texture2D ( iChannel0, uv );
   float old_luminosity = length(c.rgb);

   c = c + K + c*mat4 ( R, G, B, A );

   // Optionaly ensure that luminosity is preserved.

   if ( constant_luminosity )
    { c.rgb = c.rgb * old_luminosity / length(c.rgb);
    }

   // Clamp the result - if no clamping was requested then the
   // limits used here are effectively infinite.

   return clamp ( c, clamp_low, clamp_high );

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   uv = ( uv - hotspot ) / scale + hotspot;

   // Perform the mixing and update the shader's outputs.

   gl_FragColor = Emu_Remix_RGBA(uv) * gl_Color;

}